/*	Reporting on index needs

	This script creates and executes a stored procedure that draws information from collected index metrics in order
	to generate useful conclusions regarding index usage needs and will suggest missing indexes to add, while also
	letting you know if any indexes are underused or unused.

	This process creates a new table that is used to track recommendations and how they change over time.  This allows
	us to gauge whether a recommendation becomes more important or less important with each check.  It also lets us understand
	the effects of a change when it is implemented, which can be used to gauge success and fine-tune how we manage index
	needs going forward.
*/

IF NOT EXISTS (SELECT * FROM sys.schemas WHERE schemas.name = 'IndexMetrics')
BEGIN
	DECLARE @Schema_Creation_TSQL NVARCHAR(MAX);
	SELECT @Schema_Creation_TSQL = 'CREATE SCHEMA IndexMetrics;';
	EXEC sp_executesql @Schema_Creation_TSQL;
END
GO

IF NOT EXISTS (SELECT * FROM sys.tables WHERE tables.name = 'Index_Recommendations')
BEGIN
	CREATE TABLE IndexMetrics.Index_Recommendations
	(	Index_Recommendations_ID INT NOT NULL IDENTITY(1,1) CONSTRAINT PK_Index_Recommendations PRIMARY KEY CLUSTERED,
		Recommended_Action VARCHAR(250) NOT NULL,
		Recommended_Action_Create_Datetime DATETIME NOT NULL,
		Last_Recommended_Datetime DATETIME NOT NULL,
		Number_of_Recommendations INT NOT NULL,
		[Database_Name] SYSNAME NOT NULL,
		[Schema_Name] SYSNAME NOT NULL,
		Table_Name SYSNAME NOT NULL,
		Index_Name SYSNAME NOT NULL,
		Equality_Columns VARCHAR(MAX) NULL,
		Inequality_Columns VARCHAR(MAX) NULL,
		Include_Columns VARCHAR(MAX) NULL,
		Was_Recommendation_Implemented BIT NOT NULL,
		Recommendation_Implemented_Date DATE NULL,
		User_Seek_Count BIGINT NULL,
		User_Scan_Count BIGINT NULL,
		User_Lookup_Count BIGINT NULL,
		User_Update_Count BIGINT NULL,
		Last_User_Seek DATETIME NULL,
		Last_User_Scan DATETIME NULL,
		Last_User_Lookup DATETIME NULL,
		Last_User_Update DATETIME NULL,
		Is_Primary_Key BIT NULL,
		Is_Clustered_Index BIT NULL,
		Improvement_Measure FLOAT NULL,
		Index_Creation_Statement NVARCHAR(MAX) NULL,
		Is_Unused BIT NULL,
		Is_Dropped BIT NULL
	);
END

GO

IF EXISTS (SELECT * FROM sys.procedures WHERE procedures.name = 'Index_Recommendations_Reporting')
BEGIN
	DROP PROCEDURE IndexMetrics.Index_Recommendations_Reporting;
END
GO

/*	This stored procedure will process data from the index usage statistics and missing index tables and return
	a list of existing indexes that may require changes and a list of missing indexes that could be beneficial to add.
*/

CREATE PROCEDURE IndexMetrics.Index_Recommendations_Reporting
AS
BEGIN
	SET NOCOUNT ON;
	-- Create a generic table that will store all recommendations.
	CREATE TABLE #Index_Recommendations
	(	Index_Recommendations_ID INT NOT NULL IDENTITY(1,1) CONSTRAINT PK_Index_Recommendations PRIMARY KEY CLUSTERED,
		Recommended_Action VARCHAR(250) NOT NULL,
		Index_Metric_Creation_Datetime DATETIME NOT NULL,
		Last_Recommended_Datetime DATETIME NOT NULL,
		[Database_Name] SYSNAME NOT NULL,
		[Schema_Name] SYSNAME NOT NULL,
		Table_Name SYSNAME NOT NULL,
		Index_Name SYSNAME NOT NULL,
		Equality_Columns VARCHAR(MAX) NULL,
		Inequality_Columns VARCHAR(MAX) NULL,
		Include_Columns VARCHAR(MAX) NULL,
		User_Seek_Count BIGINT NULL,
		User_Scan_Count BIGINT NULL,
		User_Lookup_Count BIGINT NULL,
		User_Update_Count BIGINT NULL,
		Last_User_Seek DATETIME NULL,
		Last_User_Scan DATETIME NULL,
		Last_User_Lookup DATETIME NULL,
		Last_User_Update DATETIME NULL,
		Is_Primary_Key BIT NULL,
		Is_Clustered_Index BIT NULL,
		Improvement_Measure FLOAT NULL,
		Index_Creation_Statement NVARCHAR(MAX) NULL,
		Is_Unused BIT NULL,
		Is_Dropped BIT NULL	);
	-- Metrics based on index utilization data.
	-- Unused indexes 
	INSERT INTO #Index_Recommendations
		(	Recommended_Action, Index_Metric_Creation_Datetime, Last_Recommended_Datetime, [Database_Name], [Schema_Name], Table_Name, Index_Name, Equality_Columns,
			Inequality_Columns, Include_Columns, User_Seek_Count, User_Scan_Count, User_Lookup_Count, User_Update_Count, Last_User_Seek, Last_User_Scan, Last_User_Lookup,
			Last_User_Update, Is_Primary_Key, Is_Clustered_Index, Improvement_Measure, Is_Unused, Is_Dropped	)
	SELECT
		'UNUSED INDEX (DROP?)' AS Recommended_Action,
		Index_Utilization_Summary.Index_Utilization_Summary_Create_Datetime AS Index_Metric_Creation_Datetime,
		Index_Utilization_Summary.Index_Utilization_Summary_Last_Update_Datetime,
		Index_Utilization_Summary.Database_Name,
		Index_Utilization_Summary.[Schema_Name],
		Index_Utilization_Summary.Table_Name,
		Index_Utilization_Summary.Index_Name,
		NULL AS Equality_Columns,
		NULL AS Inequality_Columns,
		NULL AS Include_Columns,
		Index_Utilization_Summary.User_Seek_Count_Total,
		Index_Utilization_Summary.User_Scan_Count_Total,
		Index_Utilization_Summary.User_Lookup_Count_Total,
		Index_Utilization_Summary.User_Update_Count_Total,
		Index_Utilization_Summary.Last_User_Seek,
		Index_Utilization_Summary.Last_User_Scan,
		Index_Utilization_Summary.Last_User_Lookup,
		Index_Utilization_Summary.Last_User_Update,
		Index_Utilization_Summary.Is_Primary_Key,
		Index_Utilization_Summary.Is_Clustered_Index,
		NULL AS Improvement_Measure,
		Index_Utilization_Summary.Is_Unused,
		Index_Utilization_Summary.Is_Dropped
	FROM IndexMetrics.Index_Utilization_Summary
	WHERE Index_Utilization_Summary.Is_Unused = 1
	AND Index_Utilization_Summary.Is_Dropped = 0;
	-- Underused indexes---those that have less than 2% read operations.  2% is arbitrary and you may adjust as needed for your own applications.
	WITH CTE_UNDERUSED_INDEXES AS (
		SELECT
			*,
			CAST(Index_Utilization_Summary.User_Seek_Count_Total + Index_Utilization_Summary.User_Scan_Count_Total + Index_Utilization_Summary.User_Lookup_Count_Total AS DECIMAL(19,2)) AS Read_Count,
			CAST(Index_Utilization_Summary.User_Seek_Count_Total + Index_Utilization_Summary.User_Scan_Count_Total + Index_Utilization_Summary.User_Lookup_Count_Total + User_Update_Count_Total AS DECIMAL(19,2)) AS Operation_Count
		FROM IndexMetrics.Index_Utilization_Summary	)
	INSERT INTO #Index_Recommendations
		(	Recommended_Action, Index_Metric_Creation_Datetime, Last_Recommended_Datetime, [Database_Name], [Schema_Name], Table_Name, Index_Name, Equality_Columns,
			Inequality_Columns, Include_Columns, User_Seek_Count, User_Scan_Count, User_Lookup_Count, User_Update_Count, Last_User_Seek, Last_User_Scan, Last_User_Lookup,
			Last_User_Update, Is_Primary_Key, Is_Clustered_Index, Improvement_Measure, Is_Unused, Is_Dropped	)
	SELECT
		'UNDERUSED INDEX (DROP?)' AS Recommended_Action,
		CTE_UNDERUSED_INDEXES.Index_Utilization_Summary_Create_Datetime AS Index_Metric_Creation_Datetime,
		CTE_UNDERUSED_INDEXES.Index_Utilization_Summary_Last_Update_Datetime,
		CTE_UNDERUSED_INDEXES.Database_Name,
		CTE_UNDERUSED_INDEXES.[Schema_Name],
		CTE_UNDERUSED_INDEXES.Table_Name,
		CTE_UNDERUSED_INDEXES.Index_Name,
		NULL AS Equality_Columns,
		NULL AS Inequality_Columns,
		NULL AS Include_Columns,
		CTE_UNDERUSED_INDEXES.User_Seek_Count_Total,
		CTE_UNDERUSED_INDEXES.User_Scan_Count_Total,
		CTE_UNDERUSED_INDEXES.User_Lookup_Count_Total,
		CTE_UNDERUSED_INDEXES.User_Update_Count_Total,
		CTE_UNDERUSED_INDEXES.Last_User_Seek,
		CTE_UNDERUSED_INDEXES.Last_User_Scan,
		CTE_UNDERUSED_INDEXES.Last_User_Lookup,
		CTE_UNDERUSED_INDEXES.Last_User_Update,
		CTE_UNDERUSED_INDEXES.Is_Primary_Key,
		CTE_UNDERUSED_INDEXES.Is_Clustered_Index,
		NULL AS Improvement_Measure,
		CTE_UNDERUSED_INDEXES.Is_Unused,
		CTE_UNDERUSED_INDEXES.Is_Dropped
	FROM CTE_UNDERUSED_INDEXES
	WHERE CTE_UNDERUSED_INDEXES.Read_Count / CASE WHEN CTE_UNDERUSED_INDEXES.Operation_Count = 0 THEN 1 ELSE CTE_UNDERUSED_INDEXES.Operation_Count END > 0
	AND CTE_UNDERUSED_INDEXES.Read_Count / CASE WHEN CTE_UNDERUSED_INDEXES.Operation_Count = 0 THEN 1 ELSE CTE_UNDERUSED_INDEXES.Operation_Count END <= 0.02
	AND CTE_UNDERUSED_INDEXES.Is_Clustered_Index = 0
	AND CTE_UNDERUSED_INDEXES.Is_Primary_Key = 0
	AND CTE_UNDERUSED_INDEXES.Is_Dropped = 0;
	-- Misused indexes---Those that have 98% or more scans.  These indexes might benefit from additional sorting columns or quqery adjustment to better use them.
	-- If the index indicated here is a PK, then this indicates clustered index scans, which may indicate missing indexes that will be corroborated in the missing index metrics.
	WITH CTE_MISUSED_INDEXES AS (
		SELECT
			*,
			CAST(Index_Utilization_Summary.User_Seek_Count_Total + Index_Utilization_Summary.User_Scan_Count_Total + Index_Utilization_Summary.User_Lookup_Count_Total AS DECIMAL(19,2)) AS Read_Count
		FROM IndexMetrics.Index_Utilization_Summary	)
	INSERT INTO #Index_Recommendations
		(	Recommended_Action, Index_Metric_Creation_Datetime, Last_Recommended_Datetime, [Database_Name], [Schema_Name], Table_Name, Index_Name, Equality_Columns,
			Inequality_Columns, Include_Columns, User_Seek_Count, User_Scan_Count, User_Lookup_Count, User_Update_Count, Last_User_Seek, Last_User_Scan, Last_User_Lookup,
			Last_User_Update, Is_Primary_Key, Is_Clustered_Index, Improvement_Measure, Is_Unused, Is_Dropped	)
	SELECT
		'MISUSED INDEX (ALTER/ADJUST/CHECK MISSING INDEX RECOMMENDATIONS)' AS Recommended_Action,
		CTE_MISUSED_INDEXES.Index_Utilization_Summary_Create_Datetime AS Index_Metric_Creation_Datetime,
		CTE_MISUSED_INDEXES.Index_Utilization_Summary_Last_Update_Datetime,
		CTE_MISUSED_INDEXES.Database_Name,
		CTE_MISUSED_INDEXES.[Schema_Name],
		CTE_MISUSED_INDEXES.Table_Name,
		CTE_MISUSED_INDEXES.Index_Name,
		NULL AS Equality_Columns,
		NULL AS Inequality_Columns,
		NULL AS Include_Columns,
		CTE_MISUSED_INDEXES.User_Seek_Count_Total,
		CTE_MISUSED_INDEXES.User_Scan_Count_Total,
		CTE_MISUSED_INDEXES.User_Lookup_Count_Total,
		CTE_MISUSED_INDEXES.User_Update_Count_Total,
		CTE_MISUSED_INDEXES.Last_User_Seek,
		CTE_MISUSED_INDEXES.Last_User_Scan,
		CTE_MISUSED_INDEXES.Last_User_Lookup,
		CTE_MISUSED_INDEXES.Last_User_Update,
		CTE_MISUSED_INDEXES.Is_Primary_Key,
		CTE_MISUSED_INDEXES.Is_Clustered_Index,
		NULL AS Improvement_Measure,
		CTE_MISUSED_INDEXES.Is_Unused,
		CTE_MISUSED_INDEXES.Is_Dropped
	FROM CTE_MISUSED_INDEXES
	WHERE CTE_MISUSED_INDEXES.User_Scan_Count_Total / CASE WHEN CTE_MISUSED_INDEXES.Read_Count = 0 THEN 1 ELSE CTE_MISUSED_INDEXES.Read_Count END >= 0.98
	AND CTE_MISUSED_INDEXES.Is_Dropped = 0;

	-- Metrics based on missing index data.
	INSERT INTO #Index_Recommendations
		(	Recommended_Action, Index_Metric_Creation_Datetime, Last_Recommended_Datetime, [Database_Name], [Schema_Name], Table_Name, Index_Name, Equality_Columns,
			Inequality_Columns, Include_Columns, User_Seek_Count, User_Scan_Count, User_Lookup_Count, User_Update_Count, Last_User_Seek, Last_User_Scan, Last_User_Lookup,
			Last_User_Update, Is_Primary_Key, Is_Clustered_Index, Improvement_Measure, Is_Unused, Is_Dropped	)
	SELECT
		'CREATE MISSING INDEX' AS Recommended_Action,
		v_Missing_Index_Summary.First_Index_Suggestion_Time AS Index_Metric_Creation_Datetime,
		v_Missing_Index_Summary.Last_User_Seek AS Last_Recommended_Datetime,
		v_Missing_Index_Summary.[Database_Name],
		v_Missing_Index_Summary.[Schema_Name],
		v_Missing_Index_Summary.Table_Name,
		'Index_' + CONVERT (VARCHAR, ROW_NUMBER() OVER (ORDER BY v_Missing_Index_Summary.Index_Creation_Statement)) + '_' + v_Missing_Index_Summary.Table_Name AS Index_Name,
		v_Missing_Index_Summary.Equality_Columns,
		v_Missing_Index_Summary.Inequality_Columns,
		v_Missing_Index_Summary.Include_Columns,
		v_Missing_Index_Summary.User_Seeks AS User_Seek_Count,
		NULL AS User_Scan_Count,
		NULL AS User_Lookup_Count,
		NULL AS User_Update_Count,
		NULL AS Last_User_Seek,
		NULL AS Last_User_Scan,
		NULL AS Last_User_Lookup,
		NULL AS Last_User_Update,
		NULL AS Is_Primary_Key,
		NULL AS Is_Clustered_Index,
		v_Missing_Index_Summary.Improvement_Measure,
		NULL AS Is_Unused,
		NULL AS Is_Dropped
	FROM IndexMetrics.v_Missing_Index_Summary
	LEFT JOIN IndexMetrics.Index_Recommendations
	ON Index_Recommendations.[Database_Name] = v_Missing_Index_Summary.[Database_Name]
	AND Index_Recommendations.[Schema_Name] = v_Missing_Index_Summary.[Schema_Name]
	AND Index_Recommendations.Table_Name = v_Missing_Index_Summary.Table_Name
	AND Index_Recommendations.Equality_Columns = v_Missing_Index_Summary.Equality_Columns
	AND Index_Recommendations.Inequality_Columns = v_Missing_Index_Summary.Inequality_Columns
	AND Index_Recommendations.Include_Columns = v_Missing_Index_Summary.Include_Columns
	AND v_Missing_Index_Summary.Improvement_Measure IS NOT NULL
	WHERE v_Missing_Index_Summary.Improvement_Measure > 5
	AND (Index_Recommendations.Was_Recommendation_Implemented = 0 OR Index_Recommendations.Was_Recommendation_Implemented IS NULL); -- This is arbitrary, and intended to week out low cost/low effectiveness index suggestions.  Adjust as needed.

	-- Adjust recommendations for special scenarios.  Add additional data to be used in recommendations to make them more helpful.
	-- Clustered Indexes
	UPDATE Index_Recommendations
		SET Recommended_Action = 'NO CLUSTERED INDEX OPERATIONS.  POSSIBLE UNUSED TABLE (RESEARCH & DROP?)'
	FROM #Index_Recommendations Index_Recommendations
	WHERE Index_Recommendations.Recommended_Action = ('UNUSED INDEX (DROP?)')
	AND Index_Recommendations.Is_Clustered_Index = 1
	AND ISNULL(Index_Recommendations.Is_Unused, 0) = 1;

	-- Misused Indexes: Adjust recommendations to be more useful
	UPDATE Index_Recommendations
		SET Recommended_Action = 'HIGH CLUSTERED INDEX SCAN COUNT RATIO.  POSSIBLE MISSING INDEX, CHECK MISSING INDEX RECOMMENDATIONS AND CORRELATE TO THIS TABLE'
	FROM #Index_Recommendations Index_Recommendations
	WHERE Index_Recommendations.Recommended_Action = 'MISUSED INDEX (ALTER/ADJUST/CHECK MISSING INDEX RECOMMENDATIONS)'
	AND Index_Recommendations.Is_Clustered_Index = 1;

	-- Update the recommendations table with new data for existing indexes
	MERGE INTO IndexMetrics.Index_Recommendations AS Recommendation_Target
	USING (SELECT * FROM #Index_Recommendations WHERE Improvement_Measure IS NULL) AS Recommendation_Source
	ON (	Recommendation_Source.Recommended_Action = Recommendation_Target.Recommended_Action
		AND Recommendation_Source.[Database_Name] = Recommendation_Target.[Database_Name]
		AND Recommendation_Source.[Schema_Name] = Recommendation_Target.[Schema_Name]
		AND Recommendation_Source.Table_Name = Recommendation_Target.Table_Name
		AND Recommendation_Source.Index_Name = Recommendation_Target.Index_Name	)
	WHEN NOT MATCHED BY TARGET
	THEN INSERT
		(	Recommended_Action, Recommended_Action_Create_Datetime, Last_Recommended_Datetime, Number_of_Recommendations, [Database_Name], [Schema_Name],
			Table_Name, Index_Name, Equality_Columns, Inequality_Columns, Include_Columns, Was_Recommendation_Implemented, Recommendation_Implemented_Date,
			User_Seek_Count, User_Scan_Count, User_Lookup_Count, User_Update_Count, Last_User_Seek, Last_User_Scan, Last_User_Lookup, Last_User_Update,
			Is_Primary_Key, Is_Clustered_Index, Improvement_Measure, Index_Creation_Statement, Is_Unused, Is_Dropped	)
	VALUES	(	Recommendation_Source.Recommended_Action,
				CURRENT_TIMESTAMP,
				CURRENT_TIMESTAMP,
				1,
				Recommendation_Source.[Database_Name],
				Recommendation_Source.[Schema_Name],
				Recommendation_Source.Table_Name,
				Recommendation_Source.Index_Name,
				NULL,
				NULL,
				NULL,
				0,
				NULL,
				Recommendation_Source.User_Seek_Count,
				Recommendation_Source.User_Scan_Count,
				Recommendation_Source.User_Lookup_Count,
				Recommendation_Source.User_Update_Count,
				Recommendation_Source.Last_User_Seek,
				Recommendation_Source.Last_User_Scan,
				Recommendation_Source.Last_User_Lookup,
				Recommendation_Source.Last_User_Update,
				Recommendation_Source.Is_Primary_Key,
				Recommendation_Source.Is_Clustered_Index,
				Recommendation_Source.Improvement_Measure,
				NULL,
				Recommendation_Source.Is_Unused,
				Recommendation_Source.Is_Dropped	)
	WHEN MATCHED
	THEN UPDATE
		SET Last_Recommended_Datetime = CURRENT_TIMESTAMP,
			Number_of_Recommendations = Recommendation_Target.Number_of_Recommendations + 1,
			User_Seek_Count = Recommendation_Source.User_Seek_Count,
			User_Scan_Count = Recommendation_Source.User_Scan_Count,
			User_Lookup_Count = Recommendation_Source.User_Lookup_Count,
			User_Update_Count = Recommendation_Source.User_Update_Count,
			Last_User_Seek = Recommendation_Source.User_Seek_Count,
			Last_User_Scan = Recommendation_Source.User_Seek_Count,
			Last_User_Lookup = Recommendation_Source.User_Seek_Count,
			Last_User_Update = Recommendation_Source.User_Seek_Count;

	-- Update the recommendations table with new data for missing indexes
	MERGE INTO IndexMetrics.Index_Recommendations AS Recommendation_Target
	USING (SELECT * FROM #Index_Recommendations WHERE Improvement_Measure IS NOT NULL) AS Recommendation_Source
	ON (	Recommendation_Source.Recommended_Action = Recommendation_Target.Recommended_Action
		AND Recommendation_Source.[Database_Name] = Recommendation_Target.[Database_Name]
		AND Recommendation_Source.[Schema_Name] = Recommendation_Target.[Schema_Name]
		AND Recommendation_Source.Table_Name = Recommendation_Target.Table_Name
		AND Recommendation_Source.Equality_Columns = Recommendation_Target.Equality_Columns
		AND Recommendation_Source.Inequality_Columns = Recommendation_Target.Inequality_Columns
		AND Recommendation_Source.Include_Columns = Recommendation_Target.Include_Columns	)
	WHEN NOT MATCHED BY TARGET
	THEN INSERT
		(	Recommended_Action, Recommended_Action_Create_Datetime, Last_Recommended_Datetime, Number_of_Recommendations, [Database_Name], [Schema_Name],
			Table_Name, Index_Name, Equality_Columns, Inequality_Columns, Include_Columns, Was_Recommendation_Implemented, Recommendation_Implemented_Date,
			User_Seek_Count, User_Scan_Count, User_Lookup_Count, User_Update_Count, Last_User_Seek, Last_User_Scan, Last_User_Lookup, Last_User_Update,
			Is_Primary_Key, Is_Clustered_Index, Improvement_Measure, Index_Creation_Statement, Is_Unused, Is_Dropped	)
	VALUES	(	Recommendation_Source.Recommended_Action,
				CURRENT_TIMESTAMP,
				CURRENT_TIMESTAMP,
				1,
				Recommendation_Source.[Database_Name],
				Recommendation_Source.[Schema_Name],
				Recommendation_Source.Table_Name,
				Recommendation_Source.Index_Name,
				Recommendation_Source.Equality_Columns,
				Recommendation_Source.Inequality_Columns,
				Recommendation_Source.Include_Columns,
				0,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				NULL,
				Recommendation_Source.Improvement_Measure,
				Recommendation_Source.Index_Creation_Statement,
				NULL,
				NULL	)
	WHEN MATCHED
	THEN UPDATE
		SET Last_Recommended_Datetime = CURRENT_TIMESTAMP,
			Number_of_Recommendations = Recommendation_Target.Number_of_Recommendations + 1,
			User_Seek_Count = Recommendation_Source.User_Seek_Count,
			Improvement_Measure = Recommendation_Source.Improvement_Measure;

	-- Check to see if any index drop recommendations were implemented add that info here.  This only is used for non-aggregated data.
	UPDATE Index_Recommendations
		SET Was_Recommendation_Implemented = 1,
			Recommendation_Implemented_Date = CAST(Index_Utilization_Summary.Index_Utilization_Summary_Last_Update_Datetime AS DATE)
	FROM IndexMetrics.Index_Recommendations
	INNER JOIN IndexMetrics.Index_Utilization_Summary
	ON Index_Recommendations.[Database_Name] = Index_Utilization_Summary.[Database_Name]
	AND Index_Recommendations.[Schema_Name] = Index_Utilization_Summary.[Schema_Name]
	AND Index_Recommendations.Table_Name = Index_Utilization_Summary.Table_Name
	AND Index_Recommendations.Index_Name = Index_Utilization_Summary.Index_Name
	WHERE Index_Utilization_Summary.Is_Dropped = 1
	AND Index_Recommendations.Was_Recommendation_Implemented = 0
	AND Index_Recommendations.Improvement_Measure IS NULL
	AND Index_Recommendations.[Database_Name] <> 'ALL';;

	-- Check to see if any missing index recommendations were implemented add that info here.  This only is used for non-aggregated data.
	CREATE TABLE #Implemented_Missing_Indexes
		(	Implemented_Missing_Indexes_Id INT NOT NULL IDENTITY(1,1) PRIMARY KEY CLUSTERED,
			[Database_Name] SYSNAME NOT NULL,
			[Schema_Name] SYSNAME NOT NULL,
			Table_Name SYSNAME NOT NULL,
			Index_Column_List VARCHAR(MAX),
			Include_Columns VARCHAR(MAX)	);

	DECLARE @Sql_Command NVARCHAR(MAX) = ''
	SELECT
		@Sql_Command = @Sql_Command + '
						;WITH CTE_INDEX_COLUMNS AS (
							SELECT
								INDEX_DATA.name AS Index_Name,
								SCHEMA_DATA.name AS Schema_Name,
								TABLE_DATA.name AS Table_Name,
								INDEX_DATA.type_desc AS Index_Type,
								STUFF(( SELECT '', ['' + columns.name + '']''
										FROM [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.tables
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.schemas
										ON tables.schema_id = schemas.schema_id
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.indexes
										ON tables.object_id = indexes.object_id
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.index_columns
										ON indexes.object_id = index_columns.object_id
										AND indexes.index_id = index_columns.index_id
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.columns
										ON tables.object_id = columns.object_id
										AND index_columns.column_id = columns.column_id
										WHERE INDEX_DATA.object_id = indexes.object_id
										AND INDEX_DATA.index_id = indexes.index_id
										AND SCHEMA_DATA.schema_id = schemas.schema_id
										AND index_columns.is_included_column = 0
										AND schemas.name = ''' + CAST(Index_Recommendations.[Schema_Name] AS NVARCHAR(MAX)) + '''
										AND tables.name = ''' + CAST(Index_Recommendations.Table_Name AS NVARCHAR(MAX)) + '''
										ORDER BY index_columns.key_ordinal
									FOR XML PATH('''')), 1, 2, '''') AS Index_Column_List,
									STUFF(( SELECT '', ['' + columns.name + '']''
										FROM [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.tables
										INNER JOIN [DBTools].sys.schemas
										ON tables.schema_id = schemas.schema_id
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.indexes
										ON tables.object_id = indexes.object_id
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.index_columns
										ON indexes.object_id = index_columns.object_id
										AND indexes.index_id = index_columns.index_id
										INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.columns
										ON tables.object_id = columns.object_id
										AND index_columns.column_id = columns.column_id
										WHERE INDEX_DATA.object_id = indexes.object_id
										AND INDEX_DATA.index_id = indexes.index_id
										AND SCHEMA_DATA.schema_id = schemas.schema_id
										AND index_columns.is_included_column = 1
										AND schemas.name = ''' + CAST(Index_Recommendations.[Schema_Name] AS NVARCHAR(MAX)) + '''
										AND tables.name = ''' + CAST(Index_Recommendations.Table_Name AS NVARCHAR(MAX)) + '''
										ORDER BY index_columns.key_ordinal
									FOR XML PATH('''')), 1, 2, '''') AS Include_Column_List
							FROM [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.indexes INDEX_DATA
							INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.tables TABLE_DATA
							ON TABLE_DATA.object_id = INDEX_DATA.object_id
							INNER JOIN [' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + '].sys.schemas SCHEMA_DATA
							ON TABLE_DATA.schema_id = SCHEMA_DATA.schema_id
							WHERE SCHEMA_DATA.name = ''' + CAST(Index_Recommendations.[Schema_Name] AS NVARCHAR(MAX)) + '''
							AND TABLE_DATA.name = ''' + CAST(Index_Recommendations.Table_Name AS NVARCHAR(MAX)) + ''')
						INSERT INTO #Implemented_Missing_Indexes
							([Database_Name], [Schema_Name], Table_Name, Index_Column_List, Include_Columns)
						SELECT
							''' + CAST(Index_Recommendations.[Database_Name] AS NVARCHAR(MAX)) + ''',
							CTE_INDEX_COLUMNS.Schema_Name,
							CTE_INDEX_COLUMNS.Table_Name,
							CTE_INDEX_COLUMNS.Index_Column_List,
							ISNULL(CTE_INDEX_COLUMNS.Include_Column_List, '''') AS Include_Column_List
						FROM CTE_INDEX_COLUMNS
						WHERE CTE_INDEX_COLUMNS.Index_Type <> ''HEAP''
						AND CTE_INDEX_COLUMNS.Index_Column_List = ''' + CAST(Index_Recommendations.Equality_Columns AS NVARCHAR(MAX)) + CASE WHEN Index_Recommendations.Equality_Columns = '' OR Index_Recommendations.Inequality_Columns = '' THEN '' ELSE ', ' END + CAST(Index_Recommendations.Inequality_Columns AS NVARCHAR(MAX)) + '''
						AND CTE_INDEX_COLUMNS.Include_Column_List = ''' + CAST(Index_Recommendations.Include_Columns AS NVARCHAR(MAX)) + ''''
	FROM IndexMetrics.Index_Recommendations
	LEFT JOIN #Index_Recommendations Index_Recommendations_TEMP
	ON Index_Recommendations.[Database_Name] = Index_Recommendations_TEMP.[Database_Name]
	AND Index_Recommendations.[Schema_Name] = Index_Recommendations_TEMP.[Schema_Name]
	AND Index_Recommendations.Table_Name = Index_Recommendations_TEMP.Table_Name
	AND Index_Recommendations.Equality_Columns = Index_Recommendations_TEMP.Equality_Columns
	AND Index_Recommendations.Inequality_Columns = Index_Recommendations_TEMP.Inequality_Columns
	AND Index_Recommendations.Include_Columns = Index_Recommendations_TEMP.Include_Columns
	AND Index_Recommendations_TEMP.Improvement_Measure IS NOT NULL
	WHERE Index_Recommendations.Was_Recommendation_Implemented = 0
	AND Index_Recommendations.Improvement_Measure IS NOT NULL
	AND Index_Recommendations.[Database_Name] <> 'ALL';

	EXEC sp_executesql @Sql_Command;

	-- Does a new missing index recommendation exist for any of these?
	UPDATE Index_Recommendations
		SET Was_Recommendation_Implemented = 1,
			Recommendation_Implemented_Date = CAST(CURRENT_TIMESTAMP AS DATE)
	FROM #Implemented_Missing_Indexes Implemented_Missing_Indexes
	INNER JOIN IndexMetrics.Index_Recommendations
	ON Index_Recommendations.[Database_Name] = Implemented_Missing_Indexes.[Database_Name]
	AND Index_Recommendations.[Schema_Name] = Implemented_Missing_Indexes.[Schema_Name]
	AND Index_Recommendations.Table_Name = Implemented_Missing_Indexes.Table_Name
	AND Implemented_Missing_Indexes.Index_Column_List = 
	CAST(Index_Recommendations.Equality_Columns AS NVARCHAR(MAX)) + CASE WHEN Index_Recommendations.Equality_Columns = '' OR Index_Recommendations.Inequality_Columns = '' THEN '' ELSE ', ' END + CAST(Index_Recommendations.Inequality_Columns AS NVARCHAR(MAX))
	AND Index_Recommendations.Include_Columns = Implemented_Missing_Indexes.Include_Columns;

	DROP TABLE #Implemented_Missing_Indexes;
	DROP TABLE #Index_Recommendations;
END
GO

EXEC IndexMetrics.Populate_Index_Utilization_Data;
EXEC IndexMetrics.Populate_Missing_Index_Data;
EXEC IndexMetrics.Index_Recommendations_Reporting;
GO

SELECT
	*
FROM IndexMetrics.Index_Recommendations;
GO